/*
 * Premier import filter for avisynth
 *
 * Original code by Ben Rudiak-Gould (c) 2000
 *
 * This version fixed and extended by
 * Tom Ford <tom.ford@compsoc.net> (c) 2002
 * 
 * Thanks to Ian Roberts (for bug identification and project
 * initiation), turboronin (for initial compliation hint) and Paul
 * Roberts for being clever
 *
 * Also to MSVC and it's rather groovy assembly/source code listing
 * function
 *
 * Compiles under VC7 and VC6SP5 (tested)
 *
 * expects Premier SDK headers in ./premier-headers
 * 
 * Bug fixes:
 *
 * Pixel aspect - always square
 * Different source and destination sizes - does quick fixed point nearest neighbour resize
 * Broken colours - bitwise & 0xFF - the *(char *) type was being
 * implicitally casted to a SIGNED int before adding.
 * Memory leak/VC7 compliation problem - use Premier memory handling and cleanup correctly
 * Various doing-things-properly fixes with the AVIFile library
 *
 * Ppro Fix added by Anime2Envy All changes are labeled with A2E
 * 
 *  Licensed under the GPL
 *
 */

#include  "prSDKTypes.h" //A2E SDK added to all header for Ppro Support.
#include  "prSDKStructs.h"
#include  "prSDKPlugSuites.h"
#include  "PrSDKimport.h"

#include <vfw.h>
#include <math.h>

int ImImportImage24bpp(imStdParms* stdparms, imFileRef ref, imImportImageRec* imageRec);
int ImImportImage32bpp(imStdParms* stdparms, imFileRef ref, imImportImageRec* imageRec);


struct FileInfo {
  PAVIFILE file;
  PAVISTREAM video, audio;
  PGETFRAME getframe;
  unsigned char shift;
	unsigned char bpp;
};


int ImGetInfo(imStdParms* stdparms, imFileAccessRec* fileInfo, imFileInfoRec* imFileInfo) {

  FileInfo fi;

  //Init VFW

  AVIFileInit();

  //Set up VFW AVI file streams
  
  if (FAILED(AVIFileOpen(&fi.file, fileInfo->filespec.name, OF_READ, NULL)))
    return imBadFile;

  fi.video = 0;
  AVIFileGetStream(fi.file, &fi.video, streamtypeVIDEO, 0);
  fi.audio = 0;
  AVIFileGetStream(fi.file, &fi.audio, streamtypeAUDIO, 0);

  if (!fi.video && !fi.audio)
    return imNoContent;

  //Population Premier imFileInfo

  imFileInfo->hasVideo = false;
  imFileInfo->hasAudio = false;

  //Video
  if (fi.video) {

    imFileInfo->hasVideo = true;

    AVISTREAMINFO asi;
    if (FAILED(AVIStreamInfo(fi.video, &asi, sizeof(asi))))
      return imBadFile;

    //This is rather unusual - vidSampleSize is normally 1, maybe try scaling vidscale + duration

    imFileInfo->vidScale = int(double(asi.dwRate) * 500 / asi.dwScale + 0.5);
    imFileInfo->vidSampleSize = 500;
    imFileInfo->vidDuration = asi.dwLength * 500;
    imFileInfo->canDraw = false; //don't handle drawing ourselves

    //imImageInfoRec struct

    //Bounds info

    imFileInfo->vidInfo.imageWidth = asi.rcFrame.right;
    imFileInfo->vidInfo.imageHeight = asi.rcFrame.bottom;
    imFileInfo->vidInfo.pixelAspectNum = 1;
    imFileInfo->vidInfo.pixelAspectDen = 1;
    imFileInfo->vidInfo.depth = 32; //required by P
    imFileInfo->vidInfo.subType = imUncompressed;

    //Time info

    imFileInfo->vidInfo.noDuration = false;
    imFileInfo->vidInfo.isFramestore = false;

    //Format info

    imFileInfo->vidInfo.alphaType = alphaNone;
    imFileInfo->vidInfo.alphaInverted = false;

    //Draw info

    imFileInfo->vidInfo.fieldType = prFieldsNone;
    imFileInfo->vidInfo.fieldsStacked = false;
    imFileInfo->vidInfo.hasPulldown = false;
    imFileInfo->vidInfo.pulldownPhase = 0;
    imFileInfo->vidInfo.isVectors = false;
    imFileInfo->vidInfo.isStill = false;

    //Test single frame, return failure if cannot open (non-avisynth file)
		
    BITMAPINFOHEADER bi;
    memset(&bi, 0, sizeof(bi));
    bi.biSize = sizeof(bi);
    bi.biWidth = asi.rcFrame.right;
    bi.biHeight = asi.rcFrame.bottom;
    bi.biPlanes = 1;
    bi.biCompression = BI_RGB;

		//Prefer 32 bit bitmap, if not try 24 bit
		
		bi.biBitCount = 32;
    fi.getframe = AVIStreamGetFrameOpen(fi.video, &bi);
		if (fi.getframe) {
			fi.bpp = 32;
			//MessageBox(NULL, "32-bit", "analysis", MB_OK | MB_ICONINFORMATION);
		}
		else {
			bi.biBitCount = 24;
			//MessageBox(NULL, "24-bit", "analysis", MB_OK | MB_ICONINFORMATION);
			fi.getframe = AVIStreamGetFrameOpen(fi.video, &bi);
			if (fi.getframe)
				fi.bpp = 24;
			else {
				return imBadCodec;
			}
		}
	}

  //Audio
  if (fi.audio) {

    imFileInfo->hasAudio = true;

    AVISTREAMINFO asi;
    if (FAILED(AVIStreamInfo(fi.audio, &asi, sizeof(asi))))
      return imBadFile;

    WAVEFORMATEX wfx;
    long size = sizeof(wfx);
    if (FAILED(AVIStreamReadFormat(fi.audio, 0, &wfx, &size)))
      return imBadFile;
    imFileInfo->audInfo.audStereo = wfx.nChannels-1;
    imFileInfo->audInfo.aud16 = (wfx.wBitsPerSample>>3)-1;
    fi.shift = imFileInfo->audInfo.audStereo + imFileInfo->audInfo.aud16;
    imFileInfo->audInfo.twosComp = imFileInfo->audInfo.aud16;
    imFileInfo->audInfo.subType = imUncompressed;
    imFileInfo->audScale = wfx.nSamplesPerSec;
    imFileInfo->audSampleSize = 1;
    imFileInfo->audDuration = asi.dwLength;
  }

  // Use Premier memory handling to save FileInfo struct
  FileInfo *fileptr = (FileInfo *)stdparms->piSuites->memFuncs->newPtr(sizeof(FileInfo));
  memcpy(fileptr, &fi, sizeof(FileInfo));
  imFileInfo->privatedata = (long)fileptr; //persist VFW file struct
  
  imFileInfo->prefs = 0; //no pref dialog
  imFileInfo->hasDataRate = false; //no data rate info

  return imNoErr;
}


int ImImportImage(imStdParms* stdparms, imFileRef ref, imImportImageRec* imageRec) {

  FileInfo* pfi = (FileInfo*)imageRec->privatedata;
	if(pfi->bpp == 24)
		return ImImportImage24bpp(stdparms, ref, imageRec);
	else //pfi == 32
		return ImImportImage32bpp(stdparms, ref, imageRec);
}

int ImImportImage24bpp(imStdParms* stdparms, imFileRef ref, imImportImageRec* imageRec) {

  // get frame from VFW

  FileInfo* pfi = (FileInfo*)imageRec->privatedata;
  LPBITMAPINFOHEADER lpbi = (LPBITMAPINFOHEADER)AVIStreamGetFrame(pfi->getframe, imageRec->pos / 500);
  if (!lpbi)
    return imBadFile;

  // copy input frame into Premier's buffer

  const int src_width = lpbi->biWidth, src_height = lpbi->biHeight;
  const int dst_width = imageRec->dstWidth, dst_height = imageRec->dstHeight;
  
  // if input frame is same size as display/export frame, do a quick copy
  
  if (src_width == dst_width && src_height == dst_height) {
    
    char* src = (char*)lpbi + lpbi->biSize; //no pallette
    int* dst = (int*)imageRec->pix;
    for (int y=src_height; y; --y) {
      for (int x=0; x<src_width; ++x)
        dst[x] = (src[x*3+0] & 0xFF) + ((src[x*3+1] & 0xFF) << 8) + ((src[x*3+2] & 0xFF) << 16);
      dst += src_width;
      src += src_width*3;
    }
    return imNoErr;
  }

  // if not, do a nearest neighbour resize

  int half_dst_width = dst_width >> 1;
  int half_dst_height = dst_height >> 1;

  char *src = (char *)lpbi + lpbi->biSize;
  int *dst = (int *)imageRec->pix;

  float scaling_factor_width = src_width / (float)dst_width;
  int top_scale_width = ((int)floor(scaling_factor_width)) << 16;
  int bot_scale_width = (int)((scaling_factor_width - floor(scaling_factor_width)) * 0x10000);

  float scaling_factor_height = src_height / (float)dst_height;
  int top_scale_height = ((int)floor(scaling_factor_height)) << 16;
  int bot_scale_height = (int)((scaling_factor_height - floor(scaling_factor_height)) * 0x10000);

  int fixed_scale_width = top_scale_width | bot_scale_width;
  int fixed_scale_height = top_scale_height | bot_scale_height;

	int half_scale_width = fixed_scale_width >> 1; //start at half
	int counter_x = half_scale_width;
	int counter_y = fixed_scale_height >> 1; //start at half

	int vert_coord = counter_y >> 16;
	int horiz_coord;

	char *row_src = src;

  for (int y = dst_height; y; --y) {

    for (int x = 0; x < dst_width; ++x) {

			horiz_coord = (counter_x >> 16) * 3;
      dst[x] = (row_src[horiz_coord] & 0xFF) + ((row_src[horiz_coord + 1] & 0xFF) << 8) + ((row_src[horiz_coord + 2] & 0xFF) << 16);
      counter_x += fixed_scale_width;
    }
    
		vert_coord = counter_y >> 16;
		row_src = src + src_width * vert_coord * 3;
    counter_y += fixed_scale_height;
		counter_x = half_scale_width;

		dst += dst_width;
  }

  return imNoErr;
}

int ImImportImage32bpp(imStdParms* stdparms, imFileRef ref, imImportImageRec* imageRec) {

  // get frame from VFW

  FileInfo* pfi = (FileInfo*)imageRec->privatedata;
  LPBITMAPINFOHEADER lpbi = (LPBITMAPINFOHEADER)AVIStreamGetFrame(pfi->getframe, imageRec->pos / 500);
  if (!lpbi)
    return imBadFile;

  // copy input frame into Premier's buffer

  const int src_width = lpbi->biWidth, src_height = lpbi->biHeight;
  const int dst_width = imageRec->dstWidth, dst_height = imageRec->dstHeight;
  
  char paddingBytes = (imageRec->rowbytes) - (imageRec->dstWidth * 4); //  A2E based on code from SDK
  char* tempPix ; // A2E Same
   
  //if input frame is same size as display/export frame, do a quick copy
  
  if (src_width == dst_width && src_height == dst_height) {

		char *src = (char *)lpbi + lpbi->biSize; //no pallette
		char *dst = imageRec->pix;
		
		
		for (int y = dst_height; y; --y) { // A2E Added to makeup for the extra bytes in each row

		memcpy(dst, src, (src_width * sizeof(DWORD)));
		
		src += (dst_width * sizeof(DWORD));
		dst += (dst_width * sizeof(DWORD)+ paddingBytes);
		}


   
	return imNoErr;
  }

  // if not, do a nearest neighbour resize (fixed point)
                                                     

  
  int half_dst_width = dst_width >> 1;
  int half_dst_height = dst_height >> 1;

  DWORD *src = (DWORD *)((char *)lpbi + lpbi->biSize);
  int *dst = (int *)imageRec->pix;

  float scaling_factor_width = src_width / (float)dst_width;
  int top_scale_width = ((int)floor(scaling_factor_width)) << 16;
  int bot_scale_width = (int)((scaling_factor_width - floor(scaling_factor_width)) * 0x10000);

  float scaling_factor_height = src_height / (float)dst_height;
  int top_scale_height = ((int)floor(scaling_factor_height)) << 16;
  int bot_scale_height = (int)((scaling_factor_height - floor(scaling_factor_height)) * 0x10000);

  int fixed_scale_width = top_scale_width | bot_scale_width;
  int fixed_scale_height = top_scale_height | bot_scale_height;

	int half_scale_width = fixed_scale_width >> 1; //start at half
	int counter_x = half_scale_width;
	int counter_y = fixed_scale_height >> 1; //start at half

	int vert_coord = counter_y >> 16;
	int horiz_coord;

	DWORD *row_src = src;

  for (int y = dst_height; y; --y) {

    for (int x = 0; x < dst_width; ++x) {

			horiz_coord = counter_x >> 16;
			dst[x] = row_src[horiz_coord];
      counter_x += fixed_scale_width;
    }
    

	
	    vert_coord = counter_y >> 16;
		row_src = src + src_width * vert_coord;
    counter_y += fixed_scale_height;
		counter_x = half_scale_width ;

		tempPix = (char*)dst; // A2E Added to makeup for the extra bytes in each row.
		tempPix += paddingBytes;// A2E Same
	    dst = (int*)tempPix; //A2E Same

		dst += dst_width;
	
  }

  return imNoErr;
}


int ImImportAudio(imStdParms* stdparms, imFileRef ref, imImportAudioRec* audioRec) {

  FileInfo* pfi = (FileInfo*)audioRec->privatedata;
  long bytes = 0, samples = 0;

  if (FAILED(AVIStreamRead(pfi->audio, audioRec->sample >> pfi->shift, audioRec->size >> pfi->shift, audioRec->buffer, audioRec->size, &bytes, &samples)) || bytes != audioRec->size) {
    return imBadFile;
  }
  return imNoErr;
}


int ImInit(imStdParms* stdParms, imImportInfoRec* importInfo) {
  importInfo->canOpen = false;
  importInfo->canSave = false;
  importInfo->canDelete = false;
  importInfo->canResize = false;
  importInfo->canDoSubsize = false;
  importInfo->canDoContinuousTime = false;
  importInfo->noFile = false;
  importInfo->addToMenu = imMenuNew;
  importInfo->hasSetup = false;
  importInfo->dontCache = false;
  importInfo->setupOnDblClk = false;
  importInfo->keepLoaded = false;
  importInfo->priority = 0;
  return imNoErr;
}


int ImGetIndFormat(imStdParms* stdparms, int index, imIndFormatRec* ifrec) {
  if (index != 0)
    return imBadFormatIndex;

  ifrec->filetype = ' SVA';
  ifrec->flags = xfCanOpen | xfCanImport | xfCanReplace | xfIsMovie;
  lstrcpy(ifrec->FormatName, "Avisynth file");
  lstrcpy(ifrec->FormatShortName, "Avisynth file");
  memcpy(ifrec->PlatformExtension, "AVS\0VDR\0", 9);
  return imNoErr;
}

int ImCloseFile(imStdParms *stdParms, imFileAccessRec *SDKfileAccessRec, long privateData) {

  FileInfo *fileptr = reinterpret_cast<FileInfo *>(privateData);

  //Clean up VFW
  AVIFileRelease(fileptr->file);
  AVIFileExit();

  if (fileptr) {
    stdParms->piSuites->memFuncs->disposePtr(reinterpret_cast<char*>(fileptr));
  }
  
  return imNoErr;
}


PREMPLUGENTRY xImportEntry(int selector, imStdParms* stdParms, long param1, long param2) {
  switch (selector) {
  case imInit:
    return ImInit(stdParms, (imImportInfoRec *)param1);
  case imImportImage:
    return ImImportImage(stdParms, (imFileRef)param1, (imImportImageRec *)param2);
  case imImportAudio:
    return ImImportAudio(stdParms, (imFileRef)param1, (imImportAudioRec *)param2);
  case imGetInfo:
    return ImGetInfo(stdParms, (imFileAccessRec *)param1, (imFileInfoRec *)param2);
  case imGetIndFormat:
    return ImGetIndFormat(stdParms, param1, (imIndFormatRec *)param2);
  case imCloseFile:
    return ImCloseFile(stdParms, (imFileAccessRec *)param1, param2);
  default:
    return imNoErr;
  }
}
